<?php
$file_types = [
    'exe' => [
        'title' => 'Executable Files',
        'risk' => 'HIGH RISK',
        'description' => 'Executable files (.exe) are the most commonly used format for malware. When executed, these files can run commands, install programs, and make changes to your system without your knowledge.',
        'examples' => ['setup.exe', 'game_update.exe', 'installer.exe'],
        'tips' => [
            'Never run .exe files from untrusted sources',
            'Always scan downloaded executables with antivirus software',
            'Be wary of email attachments with .exe extensions',
            'Verify the digital signature of executable files when possible'
        ]
    ],
    'scr' => [
        'title' => 'Screensaver Files',
        'risk' => 'HIGH RISK',
        'description' => 'Screensaver files (.scr) are executable files often used to display screensavers on Windows. However, they can also execute malicious code.',
        'examples' => ['cool_screensaver.scr', 'holiday.scr', 'animation.scr'],
        'tips' => [
            'Avoid downloading screensavers from untrusted websites',
            'Treat .scr files with the same caution as .exe files',
            'Only use screensavers from the Windows Store or trusted sources',
            'Be suspicious of .scr files received via email'
        ]
    ],
    'com' => [
        'title' => 'Command Files',
        'risk' => 'HIGH RISK',
        'description' => 'Command files (.com) are similar to .exe files and are designed to run commands on your system. Malware developers use .com files because they are less scrutinized by some antivirus software.',
        'examples' => ['virus_removal.com', 'win_patch.com', 'system.com'],
        'tips' => [
            'Never run .com files from unknown sources',
            'Be especially cautious as .com files are less common today',
            'Always scan .com files with updated antivirus software',
            'Question why modern software would use this legacy format'
        ]
    ],
    'bat' => [
        'title' => 'Batch Files',
        'risk' => 'HIGH RISK',
        'description' => 'Batch files (.bat) are scripts that contain a series of commands to be executed by the command line. Attackers use them to automate malicious activities.',
        'examples' => ['update_script.bat', 'cleanup.bat', 'install.bat'],
        'tips' => [
            'Never run batch files from untrusted sources',
            'Review the contents of .bat files before execution',
            'Be wary of batch files received via email',
            'Use script blocking software when possible'
        ]
    ],
    'cmd' => [
        'title' => 'Command Scripts',
        'risk' => 'HIGH RISK',
        'description' => '.cmd files are similar to .bat files but are specifically for Windows systems. They can be used for legitimate administrative tasks but are also exploited to run harmful scripts.',
        'examples' => ['security_fix.cmd', 'update.cmd', 'patch.cmd'],
        'tips' => [
            'Only run .cmd files from trusted sources',
            'Review script contents before execution',
            'Be cautious of .cmd files claiming to fix system issues',
            'Keep Windows Defender or antivirus software updated'
        ]
    ],
    'vbs' => [
        'title' => 'VBScript Files',
        'risk' => 'HIGH RISK',
        'description' => 'VBScript files (.vbs) can run Visual Basic scripts, often used to automate tasks. Malicious actors use them to download and execute additional malware.',
        'examples' => ['invoice.vbs', 'update.vbs', 'script.vbs'],
        'tips' => [
            'Avoid running .vbs files from unknown sources',
            'Be especially cautious of .vbs files in email attachments',
            'Disable VBScript execution if not needed',
            'Use modern automation alternatives when possible'
        ]
    ],
    'js' => [
        'title' => 'JavaScript Files',
        'risk' => 'MEDIUM RISK',
        'description' => 'JavaScript files (.js) are widely used on the web but can also be used to spread malware. Clicking on a malicious .js file might result in ransomware installation or data theft.',
        'examples' => ['photo_gallery.js', 'script.js', 'update.js'],
        'tips' => [
            'Never execute .js files downloaded from untrusted sources',
            'Be cautious of JavaScript files outside of web browsers',
            'Keep browsers and security software updated',
            'Use script blocking extensions when browsing'
        ]
    ]
];

$analyzed_file = '';
$file_info = null;
$extension = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['filename'])) {
    $analyzed_file = trim($_POST['filename']);
    $extension = strtolower(pathinfo($analyzed_file, PATHINFO_EXTENSION));
    if (array_key_exists($extension, $file_types)) {
        $file_info = $file_types[$extension];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Security Analyzer</title>
    <style>
        :root {
            --accent: 136, 58, 234;
            --accent-light: 224, 204, 250;
            --accent-dark: 49, 10, 101;
        }
        body {
            font-family: system-ui, sans-serif;
            background: #13151a;
            color: white;
            line-height: 1.6;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
        }
        .analyzer {
            background: rgba(49, 10, 101, 0.33);
            padding: 2rem;
            border-radius: 8px;
            border: 1px solid rgba(224, 204, 250, 0.25);
            margin-bottom: 2rem;
        }
        .form-group {
            margin-bottom: 1rem;
        }
        input[type="text"] {
            width: 100%;
            padding: 0.5rem;
            font-size: 1rem;
            background: #1a1c23;
            border: 1px solid rgba(224, 204, 250, 0.25);
            color: white;
            border-radius: 4px;
        }
        button {
            background: rgb(136, 58, 234);
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            font-size: 1rem;
            border-radius: 4px;
            cursor: pointer;
        }
        button:hover {
            background: rgb(156, 78, 254);
        }
        .result {
            margin-top: 2rem;
            padding: 1rem;
            border-radius: 4px;
        }
        .high-risk {
            background: rgba(255, 0, 0, 0.2);
            border: 1px solid rgba(255, 0, 0, 0.3);
        }
        .medium-risk {
            background: rgba(255, 165, 0, 0.2);
            border: 1px solid rgba(255, 165, 0, 0.3);
        }
        .safe {
            background: rgba(0, 255, 0, 0.2);
            border: 1px solid rgba(0, 255, 0, 0.3);
        }
        ul {
            margin: 1rem 0;
            padding-left: 1.5rem;
        }
        h1, h2 {
            color: rgb(224, 204, 250);
        }
        .examples {
            color: #ff9494;
            font-family: monospace;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>File Security Analyzer</h1>
        
        <div class="analyzer">
            <form method="POST">
                <div class="form-group">
                    <label for="filename">Enter a filename to analyze:</label>
                    <input type="text" id="filename" name="filename" value="<?php echo htmlspecialchars($analyzed_file); ?>" placeholder="example.exe" required>
                </div>
                <button type="submit">Analyze File</button>
            </form>

            <?php if ($analyzed_file): ?>
                <div class="result <?php echo isset($file_info) ? ($file_info['risk'] === 'HIGH RISK' ? 'high-risk' : 'medium-risk') : 'safe'; ?>">
                    <h2>Analysis Result for: <?php echo htmlspecialchars($analyzed_file); ?></h2>
                    
                    <?php if ($file_info): ?>
                        <p><strong>Risk Level:</strong> <?php echo $file_info['risk']; ?></p>
                        <p><strong>File Type:</strong> <?php echo $file_info['title']; ?></p>
                        <p><?php echo $file_info['description']; ?></p>
                        
                        <h3>Common Malicious Examples:</h3>
                        <p class="examples"><?php echo implode(', ', $file_info['examples']); ?></p>
                        
                        <h3>Security Tips:</h3>
                        <ul>
                            <?php foreach ($file_info['tips'] as $tip): ?>
                                <li><?php echo htmlspecialchars($tip); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    <?php else: ?>
                        <p>This file extension (<?php echo htmlspecialchars($extension); ?>) is not in our high-risk database. However, always be cautious with files from unknown sources.</p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>

        <div class="info">
            <h2>About This Tool</h2>
            <p>This tool helps you identify potentially dangerous file types. Enter a filename (including extension) to check if it's a known risky file type and get security recommendations.</p>
            <p>Currently monitors these high-risk extensions: .exe, .scr, .com, .bat, .cmd, .vbs, .js</p>
        </div>
    </div>
</body>
</html>